/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/**
 * \file    uart.h
 *
 * \brief   uart Header File
 *
 * This header file contains the Macros definitions and Function declarations 
 * for uart test application
 *
 * \author  0043
 *
 * \version 01a,25aug,2010 Created.
 */
/*========================
 * Includes
 *========================
 */
 
#include "DM8127_types.h"
#include "DM8127_EVM.h"
#include "DM8127_uart.h"

/* ------------------------------------------------------------------------ *
 *  UART Handle                                                             *
 * ------------------------------------------------------------------------ */
typedef unsigned int UART_Handle;

/* ------------------------------------------------------------------------ *
 *  UART IDs                                                                *
 * ------------------------------------------------------------------------ */
#define UART_0                  ( 0 )
#define UART_1                  ( 1 )
#define UART_2                  ( 2 )
#define UART_3                  ( 3 )


/* ------------------------------------------------------------------------ *
 *  Prototypes                                                              *
 * ------------------------------------------------------------------------ */

/***
 * \brief: Routine to open the uart .
 *
 * \b \Descr: This routine enables the uart .This routine takes uart id  as a parameter
 * \n         which gives information about which uart to be selected  along with that
 * \n         it takes baudrate as another parameter.
 *
 *
 * \param     id   	    [IN]    uart to be opened
 *            baudrate  [IN]    uart Baudrate
 *
 *
 * \return:  uart base address on success
 * \n        0 on failure
 */
UART_Handle _UART_open
(
	UINT16 id,
	UINT32 baudrate
);
SINT16       _UART_close( UART_Handle uart );

/***
 * \brief: Routine to check if transmission is ready in uart
 *
 * \b \Descr: This routine to check if Tx Fifo is ready or not.This routine takes
 * \n         uart id as a parameter which gives information about which uart to
 * \n         be selected .
 *
 *
 * \param:    uart		    [IN]    uart base address
 *
 * \return:   0 if transmission is not ready
 *            1 if transmission i ready
 */
SINT16 _UART_xmtReady
(
	UART_Handle uart
);
/***
 * \brief:  Routine to check reception is ready in  uart
 *
 * \b \Descr: This routine checks whether RX FIFO is ready in uart.This routine
 * \n         takes uart id  as a parameter which gives information about which
 * \n         uart to be selected.
 *
 * \param     uart		    [IN]    uart base address
 *
 * \return:   1  if  reception ready
 * \n         0  if reception is not ready
 */

SINT16 _UART_rcvReady
(
	UART_Handle uart
);

/***
 * \brief: Routine to send a byte of data to uart
 *
 * \b \Descr: This routine sends one byte of data.This routine takes uart id  as a
 * \n         parameter which gives information about which uart to be selected
 * \n         along with that it contains a buffer pointer to copy the data.
 *
 * \param   uart		    [IN]    uart base address.
 *          *data         [IN]    buffer pointer to be copied.
 *
 * \return  0 on successfull  completion of  the routine.
 */


SINT16 _UART_putChar
(
	UART_Handle uart,
	UINT8 data
);
/***
 * \brief: Routine to get one byte of data
 *
 * \b \Descr: This routine to get one byte of data.This routine takes uart id  as a
 * \n         parameter which gives information about which uart to be selected along with
 * \n         that it contains a buffer pointer to copy the data.
 *
 * \param      uart		    [IN]    uart base address
 * \n          *data         [IN]    buffer pointer to copy the received data
 *
 * \return  0 on successfull  completion of  the routine.
 */

SINT16 _UART_getChar
(
	UART_Handle uart,
	char* data
);

/***
 * \brief: Routine to get the value of a uart register.
 *
 * \b \Descr: This routine get the value of a uart register.This routine takes uart
 * \n         controller base addresss as a parameter along with that register offset as
 * \n         another parameter and returns the value of the register.
 *
 *
 * \param:    uart		[IN]	uart controller base address
 *            reg         [IN]    register offset value
 *
 * \return:   returns register value
 *
 */
UINT32 _UART_rget
(
	UART_Handle uart,
	UINT32 reg
);
/***
 * \brief: Routine to set the value of a uart register.
 *
 * \b \Descr: This routine set the value of a uart register.This routine takes uart
 * \n         controller base addresss as a parameter along with that register offset as
 * \n         another parameter and the value of the register.
 *
 *
 * \param    uart		[IN]	uart controller base address
 *           reg        [IN]    register offset value
 * \return: VOID
 */
void _UART_rset
(
	UART_Handle uart,
	UINT32 reg,
	UINT32 reg_value
);
/***
 * \brief: Routine to reset the uart  conntroller.
 *
 * \b \Descr: This routine resets the uart controller.This routine takes uart
 * \n         controller base addresss as a parameter.
 *
 * \param:   uart  [IN]   uart to be resetted
 * \return:  VOID
 *
 */
void _UART_reset
(
	UART_Handle uart
);
/***
 * \brief: Routine send a byte of data to uart.
 *
 * \b \Descr: This routine send a byte of data to uart.This routine takes uart id
 * \n         as a parameter which gives information about which uart to be
 * \n         selected along with that it takes buffer pointer as another parameter
 * \n       to send  data and timeout as another parameter.
 *
 * \param   uart	    [IN]    uart to be selected
 *          tx          [IN]    Data to be transmitted
 *          timeout     [IN]    Timeout for transmitting
 *
 * \return   1 on failure
 *           0 on success
 */
SINT16 uart_send
(
	UART_Handle uart,
	UINT8 tx,
	SINT32 timeout
);

/***
 * \brief: Routine receive a byte of data from uart.
 *
 * \b \Descr: This routine receive a byte of data from uart.This routine takes uart
 * \n         id  as a parameter which gives information about which uart to be
 * \n         selected along with that it takes bufferpointer as another parameter
 * \n         to copy the received data and timeout as another parameter.
 *
 * \param     uart		 [IN]     uart to be selected
 *            *rx       [IN]     Buffer pointer to copy received data
 *            timeout      [IN]     Timeout for receiving
 *
 * \return   1 on failure
 *           0 on success
 */
SINT16 uart_recv
(
	UART_Handle uart,
	char* rx,
	SINT32 timeout
);
/***
 * \brief:  Routine to Enable Hardware Flow Control
 *
 * \param:  uart  [IN]   uart on which Hardware Flow Control needs to be
 * \n       enabled.
 * \return: SUCCESS
 */


SINT16 UART_EnableHWFlowCntl
(
	UART_Handle uart
);
/***
 * \brief: Routine to Disable Hardware Flow Control
 *
 * \param:   uart  [IN]   uart on which Hardware Flow Control needs to be
 * \n        disabled.
 * \return: SUCCESS
 */

SINT16 UART_DisableHWFlowCntl
(
	UART_Handle uart
);







